<?php
/**
 * bot.php
 * Anonymous Telegram Bot (single-file) — File-based JSON DB
 * Configured with:
 *  - Token: Line 24
 *  - Owner ID: Line 25
 *  - Base URL: Line 26
 *
 * Features:
 *  - On /start: create user folder and profile.json if missing
 *  - Step-by-step profile: gender -> age (2 pages) -> city -> region (paginated)
 *  - File-based JSON DB under /db/<tg_id>/
 *  - Main menu, Nearby, Search, Connect, accept/reject, chat, reveal, block
 *  - Admin panel: ?admin=1&owner=OWNER_ID
 *
 * Security note: rotate token after public testing if it becomes public.
 */

/* =========================
   CONFIG
   ========================= */
$CONFIG = [
  'token' => '8515698987:AAH3RDQpc_DqQdN9oAY3_zNHeIqcb6rYkB8',
  'owner_id' => 595158755,
  'base_url' => 'https://boysystem.ir',
  'data_dir' => __DIR__ . '/db',
  'users_list' => __DIR__ . '/users/user_ids.json',
  'default_coins' => 20,
  'coins_per_referral' => 10
];

/* ensure base dirs */
@mkdir($CONFIG['data_dir'], 0755, true);
@mkdir(dirname($CONFIG['users_list']), 0755, true);

/* =========================
   FILE-DB HELPERS
   ========================= */
function userFolder($tg_id) {
  global $CONFIG;
  return rtrim($CONFIG['data_dir'], '/') . '/' . intval($tg_id);
}
function ensureUserFolder($tg_id) {
  $f = userFolder($tg_id);
  if(!is_dir($f)) @mkdir($f, 0755, true);
  return $f;
}
function profilePath($tg_id) { return userFolder($tg_id) . '/profile.json'; }
function chatPath($tg_id)    { return userFolder($tg_id) . '/chat.json'; }
function pendingPath($tg_id) { return userFolder($tg_id) . '/pending.json'; }
function blocksPath($tg_id)  { return userFolder($tg_id) . '/blocks.json'; }

function saveJson($path, $data) {
  $dir = dirname($path);
  if(!is_dir($dir)) @mkdir($dir, 0755, true);
  file_put_contents($path, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
}
function loadJson($path) {
  if(!file_exists($path)) return null;
  $s = file_get_contents($path);
  return json_decode($s, true);
}

function addUserToList($tg_id) {
  global $CONFIG;
  $listFile = $CONFIG['users_list'];
  $list = [];
  if(file_exists($listFile)) $list = json_decode(file_get_contents($listFile), true) ?: [];
  if(!in_array($tg_id, $list)) {
    $list[] = $tg_id;
    file_put_contents($listFile, json_encode($list, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE), LOCK_EX);
  }
}
function allUsers() {
  global $CONFIG;
  $file = $CONFIG['users_list'];
  if(!file_exists($file)) return [];
  return json_decode(file_get_contents($file), true) ?: [];
}

/* =========================
   TELEGRAM API HELPERS
   ========================= */
function tg_api($method, $params = []) {
  global $CONFIG;
  $url = "https://api.telegram.org/bot{$CONFIG['token']}/{$method}";
  $ch = curl_init($url);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
  curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($params));
  curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
  $res = curl_exec($ch);
  if(curl_errno($ch)) error_log('cURL error: '.curl_error($ch));
  curl_close($ch);
  return json_decode($res, true);
}
function sendMessage($chat_id, $text, $extra = []) {
  $params = array_merge(['chat_id'=>$chat_id, 'text'=>$text, 'parse_mode'=>'HTML'], $extra);
  return tg_api('sendMessage', $params);
}
function editMessageText($chat_id, $message_id, $text, $extra = []) {
  $params = array_merge(['chat_id'=>$chat_id, 'message_id'=>$message_id, 'text'=>$text, 'parse_mode'=>'HTML'], $extra);
  return tg_api('editMessageText', $params);
}
function answerCallback($callback_query_id, $text = '', $show_alert = false) {
  return tg_api('answerCallbackQuery', ['callback_query_id'=>$callback_query_id, 'text'=>$text, 'show_alert'=>$show_alert]);
}
function buildInlineKeyboard($rows) { return ['reply_markup' => json_encode(['inline_keyboard' => $rows])]; }

/* =========================
   STATIC DATA: cities + main regions (Version B)
   Expand as needed
   ========================= */
$cities_regions = [
  'Tehran' => ['Narmak','Tehranpars','Vanak','Saadat Abad','Shahrak Gharb','Arg','Ekbatan','Enghelab','Pardisan','Lavasan'],
  'Mashhad' => ['Central','Vakilabad','Sajjad','Tabarsi','Shandiz'],
  'Isfahan' => ['Central','KhomeiniShahr','Najafabad','Falavarjan'],
  'Shiraz' => ['Central','Maaliabad','Sadra','Zand'],
  'Tabriz' => ['Central','Elgoli','Sahand'],
  'Karaj' => ['Central','Gohardasht','Mahdasht']
];

/* =========================
   AGE GRID: two pages (1..50, 51..100) 10x5
   ========================= */
function ageGridKeyboard($page = 1) {
  $start = ($page - 1) * 50 + 1;
  $buttons = [];
  $num = $start;
  for($r=0;$r<10;$r++){
    $row = [];
    for($c=0;$c<5;$c++){
      $label = ($num > 100) ? "OK" : (string)$num;
      $row[] = ['text'=>$label, 'callback_data'=>"age_select:{$num}:{$page}"];
      $num++;
    }
    $buttons[] = $row;
  }
  $nav = [];
  if($page > 1) $nav[] = ['text'=>'⬅️ Prev','callback_data'=>'age_page:'.($page-1)];
  $nav[] = ['text'=>"Page {$page}/2",'callback_data'=>'age_page_info'];
  if($page < 2) $nav[] = ['text'=>'Next ➡️','callback_data'=>'age_page:'.($page+1)];
  $buttons[] = $nav;
  return $buttons;
}

/* =========================
   PROFILE create/load/save
   ========================= */
function createOrGetProfile($from) {
  $tg_id = $from['id'];
  ensureUserFolder($tg_id);
  $profile = loadJson(profilePath($tg_id));
  if(!$profile) {
    $profile = [
      'tg_id' => $tg_id,
      'username' => $from['username'] ?? null,
      'first_name' => $from['first_name'] ?? null,
      'last_name' => $from['last_name'] ?? null,
      'gender' => null,
      'age' => null,
      'city' => null,
      'region' => null,
      'created_at' => date('c'),
      'coins' => $GLOBALS['CONFIG']['default_coins'],
      'anonymous_link_token' => bin2hex(random_bytes(6)),
      'profile_complete' => false
    ];
    saveJson(profilePath($tg_id), $profile);
    addUserToList($tg_id);
  }
  return $profile;
}
function saveProfile($tg_id, $profile) { saveJson(profilePath($tg_id), $profile); }

/* =========================
   CHAT helpers
   ========================= */
function startChatBetween($a,$b) {
  $time = date('c');
  saveJson(userFolder($a).'/chat.json', ['active'=>true,'with'=>$b,'started_at'=>$time]);
  saveJson(userFolder($b).'/chat.json', ['active'=>true,'with'=>$a,'started_at'=>$time]);
}
function endChatFor($tg_id) {
  $chat = loadJson(userFolder($tg_id).'/chat.json');
  if(!$chat) return;
  $partner = $chat['with'] ?? null;
  if($partner) saveJson(userFolder($partner).'/chat.json', ['active'=>false,'with'=>null,'ended_at'=>date('c')]);
  saveJson(userFolder($tg_id).'/chat.json', ['active'=>false,'with'=>null,'ended_at'=>date('c')]);
}
function isInActiveChat($tg_id) {
  $c = loadJson(userFolder($tg_id).'/chat.json');
  return ($c && !empty($c['active']));
}
function getChatPartner($tg_id) {
  $c = loadJson(userFolder($tg_id).'/chat.json');
  return $c ? ($c['with'] ?? null) : null;
}

/* Blocks */
function blockUser($blocker, $blocked) {
  $path = blocksPath($blocker);
  $arr = loadJson($path) ?: [];
  if(!in_array($blocked, $arr)) {
    $arr[] = $blocked;
    saveJson($path, $arr);
  }
}
function isBlockedBy($user, $byUser) {
  $arr = loadJson(blocksPath($byUser)) ?: [];
  return in_array($user, $arr);
}

/* =========================
   MENUS & PROMPTS
   ========================= */
function showWelcome($chat_id) {
  $text = "👋 Welcome to <b>Anonymous Chat</b>\nPlease choose your gender to continue:";
  $buttons = [
    [['text'=>"👦 I'm a boy",'callback_data'=>"set_gender:male"], ['text'=>"👧 I'm a girl",'callback_data'=>"set_gender:female"]]
  ];
  return sendMessage($chat_id, $text, buildInlineKeyboard($buttons));
}
function askAgePage($chat_id, $page=1) {
  return sendMessage($chat_id, "🔢 Choose your age (Page {$page}):", buildInlineKeyboard(ageGridKeyboard($page)));
}
function askCity($chat_id) {
  global $cities_regions;
  $rows = []; $row = []; $i=0;
  foreach(array_keys($cities_regions) as $city) {
    $row[] = ['text'=>$city, 'callback_data'=>"city_select:".rawurlencode($city)];
    $i++;
    if($i % 2 == 0) { $rows[] = $row; $row = []; }
  }
  if(count($row)) $rows[] = $row;
  $rows[] = [['text'=>'🌐 Other (type city name)', 'callback_data'=>'city_other']];
  return sendMessage($chat_id, "🏙️ Choose your city:", buildInlineKeyboard($rows));
}
function askRegionPage($chat_id, $city, $page=1, $per_page=8) {
  global $cities_regions;
  $list = $cities_regions[$city] ?? [];
  $total = count($list);
  $start = ($page-1)*$per_page;
  $slice = array_slice($list, $start, $per_page);
  $rows = []; $row = [];
  foreach($slice as $r) {
    $row[] = ['text'=>$r, 'callback_data'=>"region_select:".rawurlencode($city).":".rawurlencode($r).":{$page}"];
    if(count($row) === 2) { $rows[] = $row; $row = []; }
  }
  if(count($row)) $rows[] = $row;
  $nav = [];
  if($page > 1) $nav[] = ['text'=>'⬅️ Prev','callback_data'=>"region_page:".rawurlencode($city).":".($page-1)];
  $nav[] = ['text'=>"Page {$page}", 'callback_data'=>"region_page_info"];
  if($start + $per_page < $total) $nav[] = ['text'=>'Next ➡️','callback_data'=>"region_page:".rawurlencode($city).":".($page+1)];
  $rows[] = $nav;
  $rows[] = [['text'=>'🌐 Other (type region)', 'callback_data'=>"region_other:".rawurlencode($city)]];
  return sendMessage($chat_id, "📍 Choose your region in <b>{$city}</b>:", buildInlineKeyboard($rows));
}
function mainMenuKeyboard() {
  return [
    [['text'=>"🔗 Connect me to a stranger",'callback_data'=>"connect_random"]],
    [['text'=>"📍 Nearby users",'callback_data'=>"nearby_search"], ['text'=>"🔎 Search users",'callback_data'=>"search_users"]],
    [['text'=>"❓ Help",'callback_data'=>"help"], ['text'=>"👤 Profile",'callback_data'=>"profile"], ['text'=>"💰 Coins",'callback_data'=>"coins"]],
    [['text'=>"🤝 Invite friends (Referral)",'callback_data'=>"referral"]],
    [['text'=>"🔗 My anonymous link",'callback_data'=>"my_anon_link"]]
  ];
}
function showMainMenu($chat_id) {
  return sendMessage($chat_id, "🏠 Main Menu — choose an option:", buildInlineKeyboard(mainMenuKeyboard()));
}

/* =========================
   READ UPDATE
   ========================= */
$raw = file_get_contents('php://input');
$update = json_decode($raw, true);

/* ADMIN PANEL via browser */
if(isset($_GET['admin'])) {
  $owner = $CONFIG['owner_id'];
  if(!isset($_GET['owner']) || intval($_GET['owner']) !== $owner) {
    echo "<h3>Admin Panel</h3><p>Use ?admin=1&owner={$owner}</p>";
    exit;
  }
  echo "<h2>Admin Panel</h2>";
  $users = allUsers();
  echo "<p>Total users: ".count($users)."</p>";
  echo "<h3>Users IDs</h3><pre>".htmlspecialchars(json_encode($users, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))."</pre>";
  echo "<h3>Broadcast</h3><form method='post'><textarea name='msg' cols='60' rows='6'></textarea><br><input type='submit' value='Send Broadcast'></form>";
  if($_SERVER['REQUEST_METHOD'] === 'POST' && !empty($_POST['msg'])) {
    $msg = trim($_POST['msg']); $count=0;
    foreach($users as $u) { sendMessage($u, "[ADMIN BROADCAST]\n".$msg); $count++; }
    echo "<p>Broadcast sent to {$count} users.</p>";
  }
  exit;
}

/* if no update */
if(!$update) { http_response_code(200); echo "OK - Anonymous Bot"; exit; }

/* ensure profile exists */
function ensureProfileForFrom($from) { $p = createOrGetProfile($from); addUserToList($from['id']); return $p; }

/* MESSAGE HANDLING */
if(isset($update['message'])) {
  $msg = $update['message'];
  $from = $msg['from'];
  $chat_id = $msg['chat']['id'];
  $profile = ensureProfileForFrom($from);

  // handle /start
  if(isset($msg['text']) && strpos(trim($msg['text']), '/start') === 0) {
    // referral token
    $parts = explode(' ', trim($msg['text']));
    if(isset($parts[1])) {
      $token = $parts[1];
      foreach(allUsers() as $uid) {
        $p = loadJson(userFolder($uid).'/profile.json');
        if($p && isset($p['anonymous_link_token']) && $p['anonymous_link_token'] === $token && $p['tg_id'] != $from['id']) {
          $p['coins'] = intval($p['coins'] ?? 0) + $GLOBALS['CONFIG']['coins_per_referral'];
          saveJson(userFolder($uid).'/profile.json', $p);
          break;
        }
      }
    }
    // ensure folder & profile exist (bootstrap)
    ensureUserFolder($from['id']);
    if(!file_exists(profilePath($from['id']))) { $profile = createOrGetProfile($from); }
    else { $profile = loadJson(profilePath($from['id'])); }
    showWelcome($chat_id);
    exit;
  }

  // messages while in active chat -> forward
  if(isInActiveChat($from['id']) && isset($msg['text'])) {
    $text = $msg['text'];
    $partner = getChatPartner($from['id']);
    if(!$partner) { sendMessage($chat_id, "Error: partner not found"); exit; }
    if(isBlockedBy($from['id'], $partner)) { sendMessage($chat_id, "⚠️ You are blocked by the other user. Chat ended."); endChatFor($from['id']); exit; }
    if($text === '/leave') { endChatFor($from['id']); sendMessage($chat_id, "You left the chat."); sendMessage($partner, "⚠️ The anonymous has left the chat."); exit; }
    if($text === '/reveal') {
      $buttons = [[ ['text'=>"🔓 Accept Reveal",'callback_data'=>"accept_reveal:{$from['id']}"], ['text'=>"🚫 Reject Reveal",'callback_data'=>"reject_reveal:{$from['id']}"] ]];
      sendMessage($partner, "🔒 The anonymous requests to reveal identity. Accept to share profile.", buildInlineKeyboard($buttons));
      sendMessage($chat_id, "Reveal request sent. Waiting for partner.");
      exit;
    }
    if($text === '/block') { blockUser($from['id'], $partner); endChatFor($from['id']); sendMessage($chat_id, "User blocked and chat ended."); sendMessage($partner, "⚠️ You were blocked by the anonymous user."); exit; }
    sendMessage($partner, "<b>Anonymous</b>: " . htmlspecialchars($msg['text']));
    exit;
  }

  // pending expectations (city_other, region_other, change_anon_name)
  $pending = loadJson(pendingPath($from['id'])) ?: [];
  if(!empty($pending['expect']) && isset($msg['text'])) {
    $expect = $pending['expect']; $text = trim($msg['text']);
    if($expect === 'city_other') {
      $profile['city'] = $text; saveProfile($from['id'], $profile); @unlink(pendingPath($from['id']));
      sendMessage($chat_id, "City saved as <b>{$profile['city']}</b>. Now type your region (or send 'skip'):");
      saveJson(pendingPath($from['id']), ['expect'=>'region_other','city'=>$profile['city']]);
      exit;
    }
    if($expect === 'region_other') {
      if(strtolower($text) === 'skip') { $profile['region'] = null; } else { $profile['region'] = $text; }
      $profile['profile_complete'] = true; saveProfile($from['id'], $profile); @unlink(pendingPath($from['id']));
      sendMessage($chat_id, "✅ Profile complete."); showMainMenu($chat_id); exit;
    }
    if($expect === 'change_anon_name') {
      $profile['anonymous_name'] = $text; saveProfile($from['id'], $profile); @unlink(pendingPath($from['id']));
      sendMessage($chat_id, "Anonymous name updated."); exit;
    }
  }

  // default text handlers outside chat
  if(isset($msg['text'])) {
    $text = trim($msg['text']);
    if(strtolower($text) === 'menu' || $text === '/menu') { showMainMenu($chat_id); exit; }
    // search filters: gender,age,city,region?
    if(preg_match('/^(male|female)\s*,\s*(\d{1,3})\s*,\s*([^,]+)(?:\s*,\s*(.+))?$/i', $text, $m)) {
      $filters = ['gender'=>strtolower($m[1]), 'age'=>intval($m[2]), 'city'=>trim($m[3])];
      if(isset($m[4]) && strlen(trim($m[4]))>0) $filters['region']=trim($m[4]);
      $candidate = null;
      foreach(allUsers() as $uid) {
        if($uid == $from['id']) continue;
        $p = loadJson(userFolder($uid).'/profile.json'); if(!$p) continue;
        if(empty($p['profile_complete'])) continue;
        if(!empty($filters['gender']) && strcasecmp($p['gender'],$filters['gender'])!==0) continue;
        if(!empty($filters['age']) && intval($p['age']) !== intval($filters['age'])) continue;
        if(!empty($filters['city']) && strcasecmp($p['city'],$filters['city'])!==0) continue;
        if(!empty($filters['region']) && strcasecmp($p['region'],$filters['region'])!==0) continue;
        $chat = loadJson(userFolder($uid).'/chat.json'); if($chat && !empty($chat['active'])) continue;
        $candidate = $p; break;
      }
      if(!$candidate) { sendMessage($chat_id, "No user found matching your filters."); exit; }
      $buttons = [[ ['text'=>"✅ Accept",'callback_data'=>"accept_conn:{$from['id']}"], ['text'=>"❌ Reject",'callback_data'=>"reject_conn:{$from['id']}"] ]];
      sendMessage($candidate['tg_id'], "🕵️ A user matching your search wants to connect. Accept?", buildInlineKeyboard($buttons));
      sendMessage($chat_id, "Request sent. Waiting for acceptance."); exit;
    }
    sendMessage($chat_id, "I didn't understand. Use <b>menu</b> or the buttons."); exit;
  }
}

/* CALLBACK handling */
if(isset($update['callback_query'])) {
  $cb = $update['callback_query'];
  $data = $cb['data'];
  $from = $cb['from'];
  $msg = $cb['message'];
  $chat_id = $msg['chat']['id'];
  $msg_id = $msg['message_id'];
  $profile = createOrGetProfile($from);

  // set gender
  if(strpos($data, 'set_gender:') === 0) {
    $g = explode(':',$data,2)[1];
    $profile['gender'] = ($g === 'male') ? 'male' : 'female';
    saveProfile($from['id'], $profile);
    answerCallback($cb['id'], "Gender set to {$g}");
    askAgePage($from['id'], 1);
    exit;
  }

  // age paging
  if(strpos($data, 'age_page:') === 0) {
    $page = intval(explode(':',$data,2)[1]);
    editMessageText($chat_id, $msg_id, "🔢 Choose your age (Page {$page}):", buildInlineKeyboard(ageGridKeyboard($page)));
    exit;
  }
  if($data === 'age_page_info') { answerCallback($cb['id'], "Use Next / Prev to navigate age pages."); exit; }

  // age selected
  if(strpos($data, 'age_select:') === 0) {
    $parts = explode(':',$data);
    $age = intval($parts[1]);
    $profile['age'] = $age; saveProfile($from['id'], $profile);
    answerCallback($cb['id'], "Age set: {$age}");
    askCity($from['id']);
    exit;
  }

  // city selected
  if(strpos($data, 'city_select:') === 0) {
    $city = rawurldecode(explode(':',$data,2)[1]);
    $profile['city'] = $city; saveProfile($from['id'], $profile);
    askRegionPage($from['id'], $city, 1); exit;
  }
  if($data === 'city_other') {
    saveJson(pendingPath($from['id']), ['expect'=>'city_other']); answerCallback($cb['id'], "Please type your city name."); sendMessage($from['id'], "Please type your city name:"); exit;
  }

  // region pagination
  if(strpos($data, 'region_page:') === 0) {
    $parts = explode(':',$data);
    $city = rawurldecode($parts[1]); $page = intval($parts[2]);
    askRegionPage($from['id'], $city, $page); answerCallback($cb['id'], "Page {$page}"); exit;
  }

  // region selected
  if(strpos($data, 'region_select:') === 0) {
    $parts = explode(':',$data,4);
    $city = rawurldecode($parts[1]); $region = rawurldecode($parts[2]);
    $profile['city'] = $city; $profile['region'] = $region; $profile['profile_complete'] = true; saveProfile($from['id'], $profile);
    answerCallback($cb['id'], "Region set: {$region}");
    sendMessage($from['id'], "✅ Profile complete.\nName: <b>".htmlspecialchars($profile['first_name'] ?? '—')."</b>\nGender: <b>{$profile['gender']}</b>\nAge: <b>{$profile['age']}</b>\nCity: <b>{$profile['city']}</b>\nRegion: <b>{$profile['region']}</b>");
    showMainMenu($from['id']); exit;
  }
  if(strpos($data, 'region_other:') === 0) {
    $city = rawurldecode(explode(':',$data,2)[1]);
    saveJson(pendingPath($from['id']), ['expect'=>'region_other','city'=>$city]); answerCallback($cb['id'], "Please type your region/area name."); sendMessage($from['id'], "Please type your region/area name in <b>{$city}</b>:"); exit;
  }

  // main menu actions
  if($data === 'connect_random') {
    if(empty($profile['profile_complete'])) { answerCallback($cb['id'], "Complete profile first."); sendMessage($from['id'], "Please complete your profile."); exit; }
    $cands = [];
    foreach(allUsers() as $uid) {
      if($uid == $from['id']) continue;
      $p = loadJson(userFolder($uid).'/profile.json'); if(!$p) continue;
      if(empty($p['profile_complete'])) continue;
      if(isset($profile['gender']) && isset($p['gender']) && $profile['gender'] === $p['gender']) continue;
      if(!empty($profile['age']) && !empty($p['age']) && abs(intval($profile['age']) - intval($p['age'])) > 3) continue;
      if(!empty($profile['city']) && strcasecmp($profile['city'],$p['city'])!==0) continue;
      $chat = loadJson(userFolder($uid).'/chat.json'); if($chat && !empty($chat['active'])) continue;
      $cands[] = $p;
    }
    if(empty($cands)) { answerCallback($cb['id'], "No matches found."); sendMessage($from['id'], "No matches found right now."); exit; }
    $cand = $cands[array_rand($cands)];
    $buttons = [[ ['text'=>"✅ Accept",'callback_data'=>"accept_conn:{$from['id']}"], ['text'=>"❌ Reject",'callback_data'=>"reject_conn:{$from['id']}"] ]];
    sendMessage($cand['tg_id'], "🕵️ Anonymous connection request:\nA user matching your filters wants to connect. Accept?", buildInlineKeyboard($buttons));
    sendMessage($from['id'], "Request sent. Waiting for acceptance..."); answerCallback($cb['id'], "Request sent."); exit;
  }

  if($data === 'nearby_search') {
    if(empty($profile['profile_complete'])) { answerCallback($cb['id'], "Complete profile first."); sendMessage($from['id'], "Please complete your profile first."); exit; }
    $cands = [];
    foreach(allUsers() as $uid) {
      if($uid == $from['id']) continue;
      $p = loadJson(userFolder($uid).'/profile.json'); if(!$p) continue;
      if(empty($p['profile_complete'])) continue;
      if(isset($profile['gender']) && isset($p['gender']) && $profile['gender'] === $p['gender']) continue;
      if(!empty($profile['age']) && !empty($p['age']) && abs(intval($profile['age']) - intval($p['age'])) > 3) continue;
      if(!empty($profile['city']) && strcasecmp($profile['city'],$p['city'])!==0) continue;
      $chat = loadJson(userFolder($uid).'/chat.json'); if($chat && !empty($chat['active'])) continue;
      $cands[] = $p;
    }
    if(empty($cands)) { sendMessage($from['id'], "No nearby users found."); exit; }
    $buttons = []; $text = "Nearby users:"; $count = 0;
    foreach($cands as $c) {
      if($count++ >= 3) break;
      $text .= "\n• Anon (age: {$c['age']}, region: ".htmlspecialchars($c['region'] ?? '—').")";
      $buttons[] = [['text'=>"Connect to Anon ({$c['age']})",'callback_data'=>"request_connect:".$c['tg_id']]];
    }
    sendMessage($from['id'], $text, buildInlineKeyboard($buttons)); exit;
  }

  if(strpos($data, 'request_connect:') === 0) {
    $target = intval(explode(':',$data,2)[1]);
    $buttons = [[ ['text'=>"✅ Accept",'callback_data'=>"accept_conn:{$from['id']}"], ['text'=>"❌ Reject",'callback_data'=>"reject_conn:{$from['id']}"] ]];
    sendMessage($target, "🕵️ A nearby user wants to connect. Accept?", buildInlineKeyboard($buttons));
    sendMessage($from['id'], "Request sent."); answerCallback($cb['id'], "Request forwarded."); exit;
  }

  if($data === 'search_users') { sendMessage($from['id'], "🔎 Send search filters as: gender,age,city,region (region optional). Example: female,25,Tehran,District 3"); answerCallback($cb['id'], "Send filters as text."); exit; }

  if($data === 'help') {
    $text = "❓ Guide:\n• Use Connect to match with opposite-gender users.\n• Nearby uses your profile (city/region/age).\n• Chat commands: /leave, /reveal, /block.\n• Use My anonymous link to invite others.";
    editMessageText($chat_id, $msg_id, $text); exit;
  }

  if($data === 'profile') {
    $anon = $profile['anonymous_link_token'] ?? bin2hex(random_bytes(6));
    $profile['anonymous_link_token'] = $anon; saveProfile($from['id'], $profile);
    $link = $CONFIG['base_url'] . '/?start=' . $anon;
    $text = "👤 Profile:\nName: <b>".htmlspecialchars($profile['first_name'] ?? '—')."</b>\nGender: ".htmlspecialchars($profile['gender'] ?? '—')."\nAge: ".htmlspecialchars($profile['age'] ?? '—')."\nCity: ".htmlspecialchars($profile['city'] ?? '—')."\nRegion: ".htmlspecialchars($profile['region'] ?? '—')."\nCoins: ".intval($profile['coins'])."\nAnon link: <code>{$link}</code>";
    $buttons = [[ ['text'=>"✏️ Change anon name",'callback_data'=>"change_anon_name"], ['text'=>"🔗 Reset anon link",'callback_data'=>"reset_anon_link"] ]];
    editMessageText($chat_id, $msg_id, $text, buildInlineKeyboard($buttons)); exit;
  }

  if($data === 'coins') { $text = "💰 Coins: ".intval($profile['coins'])."\nEarn by referrals."; editMessageText($chat_id, $msg_id, $text); exit; }

  if($data === 'referral') {
    $token = $profile['anonymous_link_token'] ?? bin2hex(random_bytes(6));
    $profile['anonymous_link_token'] = $token; saveProfile($from['id'], $profile);
    $link = $CONFIG['base_url'] . '/?start=' . $token;
    editMessageText($chat_id, $msg_id, "🤝 Your referral link:\n{$link}\nEach sign-up gives you {$CONFIG['coins_per_referral']} coins."); exit;
  }

  if($data === 'my_anon_link') {
    $token = $profile['anonymous_link_token'] ?? bin2hex(random_bytes(6));
    $profile['anonymous_link_token'] = $token; saveProfile($from['id'], $profile);
    $link = $CONFIG['base_url'] . '/?start=' . $token;
    editMessageText($chat_id, $msg_id, "🔗 Your anonymous link:\n{$link}"); exit;
  }

  if(strpos($data, 'accept_conn:') === 0) {
    $requester = intval(explode(':',$data,2)[1]);
    if(isInActiveChat($from['id']) || isInActiveChat($requester)) { answerCallback($cb['id'], "Either you or requester is busy."); exit; }
    startChatBetween($requester, $from['id']);
    sendMessage($requester, "✅ Connection accepted! You are now anonymously connected.");
    sendMessage($from['id'], "✅ You accepted! You are now anonymously connected.");
    answerCallback($cb['id'], "Chat started."); exit;
  }
  if(strpos($data, 'reject_conn:') === 0) {
    $requester = intval(explode(':',$data,2)[1]);
    sendMessage($requester, "❌ Your connection request was rejected."); answerCallback($cb['id'], "Rejected."); exit;
  }

  if(strpos($data, 'accept_reveal:') === 0) {
    $req = intval(explode(':',$data,2)[1]);
    $me = loadJson(profilePath($from['id'])); $them = loadJson(profilePath($req));
    if($me && $them) {
      $p1 = "Profile Reveal:\nUsername: ".($me['username'] ?? '—')."\nName: ".($me['first_name'] ?? '—')."\nID: {$me['tg_id']}";
      $p2 = "Profile Reveal:\nUsername: ".($them['username'] ?? '—')."\nName: ".($them['first_name'] ?? '—')."\nID: {$them['tg_id']}";
      sendMessage($req, $p1); sendMessage($from['id'], $p2); answerCallback($cb['id'], "Profiles shared.");
    } else answerCallback($cb['id'], "Error reading profiles.");
    exit;
  }
  if(strpos($data, 'reject_reveal:') === 0) {
    $req = intval(explode(':',$data,2)[1]); sendMessage($req, "🔒 Reveal request was rejected."); answerCallback($cb['id'], "Rejected."); exit;
  }

  if($data === 'change_anon_name') {
    saveJson(pendingPath($from['id']), ['expect'=>'change_anon_name']); sendMessage($from['id'], "Send the new anonymous name (example: Anon#42)."); answerCallback($cb['id'], "Please send new anonymous name."); exit;
  }
  if($data === 'reset_anon_link') {
    $token = bin2hex(random_bytes(6)); $profile['anonymous_link_token'] = $token; saveProfile($from['id'], $profile);
    editMessageText($chat_id, $msg_id, "✅ Anon link reset:\n{$CONFIG['base_url']}/?start={$token}"); exit;
  }

  answerCallback($cb['id'], "Action received.");
}

/* END */
http_response_code(200);
echo 'OK';