<?php
/**
 * Instagram Downloader Telegram Bot - rewritten & debugged
 * version: 3.2.0-rewrite
 * - keeps original behavior & admin panel
 * - adds: force-join checks, optional channels, txt/json storage toggle, robust logging
 *
 * IMPORTANT:
 *  - Edit the CONFIG section below (BOT_TOKEN, API_KEY, FORCE_JOIN, CHANNELS, USE_TXT_FILES)
 *  - Ensure webserver user can write files in this dir (users/status/admin_state/debug.log)
 *  - Check webhook via: https://api.telegram.org/bot<YOUR_TOKEN>/getWebhookInfo
 */

/* ============================
   CONFIG — ویرایش کن قبل از آپلود
   ============================ */
define('BOT_TOKEN', '8339306145:AAFAEkNWyhZyTTRjjEHWBzplYUR0gFiHXYA');
define('API_KEY', '914591370:H1dkfGOXgFJRrQK@Api_ManagerRoBot');
define('INSTAGRAM_API_URL', 'https://api.fast-creat.ir/instagram');
define('TELEGRAM_API_URL', 'https://api.telegram.org/bot' . BOT_TOKEN . '/');
define('ADMIN_ID', 595158755);

// Force join: اگر true باشه کاربر باید عضو کانال‌های REQUIRED_CHANNELS باشه تا ربات کار کنه
define('FORCE_JOIN', true);

// REQUIRED_CHANNELS: اگر FORCE_JOIN=true، لیست کانال/گروه‌هایی که عضویت اجباری است.
// هر آیتم می‌تواند نام کاربری کانال بدون @ (مثلا "mychannel") یا آیدی عددی - حداکثر چند آیتم بذار
$REQUIRED_CHANNELS = [
    // 'example_channel1', 'example_channel2'
];

// OPTIONAL_CHANNELS: اگر FORCE_JOIN=false، حداکثر تا 3 کانال که دکمه‌شون نمایش داده میشه
$OPTIONAL_CHANNELS = [
    // 'example_channel_a', 'example_channel_b'
];

// Storage mode: true => use simple TXT files auto-created, false => use JSON files (original behavior)
define('USE_TXT_FILES', true);

// Filenames (will be created if not exist)
define('USERS_FILE_JSON', 'users.json');
define('STATUS_FILE_JSON', 'status.json');
define('STATE_FILE_JSON', 'admin_state.json');
define('USERS_FILE_TXT', 'users.txt');          // format: each line "user_id|join_time"
define('STATUS_FILE_TXT', 'status.txt');        // format: "bot_active:1" or "bot_active:0"
define('STATE_FILE_TXT', 'admin_state.txt');    // format: each line JSON per admin entry

// Debug: log file (ensure writable)
define('DEBUG_LOG', __DIR__ . '/debug.log');

/* ============================
   END CONFIG
   ============================ */

// enable debug display temporarily (turn off in production if you want)
ini_set('display_errors', 0);
ini_set('display_startup_errors', 0);
error_reporting(E_ALL);

function debug_log($text) {
    $line = date('[Y-m-d H:i:s] ') . $text . PHP_EOL;
    @file_put_contents(DEBUG_LOG, $line, FILE_APPEND | LOCK_EX);
}

/* ---------- TELEGRAM REQUEST ---------- */
function telegram_request($method, $params = []) {
    $ch = curl_init();
    $url = TELEGRAM_API_URL . $method;
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    // Telegram expects JSON for many endpoints
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);

    // sendMediaGroup expects 'media' as JSON string
    if ($method === 'sendMediaGroup' && isset($params['media']) && is_array($params['media'])) {
        $params['media'] = json_encode($params['media'], JSON_UNESCAPED_UNICODE);
    }

    $payload = json_encode($params, JSON_UNESCAPED_UNICODE);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);

    debug_log("telegram_request: $method payload=" . $payload);

    $resp = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if (curl_errno($ch)) {
        $err = curl_error($ch);
        debug_log("telegram_request cURL error: $err http=$http");
        curl_close($ch);
        return ['ok' => false, 'error' => $err, 'http_code' => $http];
    }
    curl_close($ch);

    debug_log("telegram_request resp_http=$http resp_body=" . $resp);
    $dec = json_decode($resp, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        debug_log("telegram_request json decode error: " . json_last_error_msg());
        return ['ok' => false, 'error' => 'invalid json response', 'raw' => $resp];
    }
    return $dec;
}

function sendMessage($chat_id, $text, $keyboard = null, $reply_to_message_id = null) {
    $params = ['chat_id' => $chat_id, 'text' => $text, 'parse_mode' => 'Markdown', 'disable_web_page_preview' => true];
    if ($keyboard) $params['reply_markup'] = $keyboard;
    if ($reply_to_message_id) $params['reply_to_message_id'] = $reply_to_message_id;
    return telegram_request('sendMessage', $params);
}

function deleteMessage($chat_id, $message_id) {
    return telegram_request('deleteMessage', ['chat_id' => $chat_id, 'message_id' => $message_id]);
}

function sendMediaGroup($chat_id, $media, $reply_to_message_id = null) {
    $params = ['chat_id' => $chat_id, 'media' => $media];
    if ($reply_to_message_id) $params['reply_to_message_id'] = $reply_to_message_id;
    return telegram_request('sendMediaGroup', $params);
}

function sendPhoto($chat_id, $photo, $caption = null, $reply_to_message_id = null, $keyboard = null) {
    $params = ['chat_id' => $chat_id, 'photo' => $photo, 'caption' => $caption, 'parse_mode' => 'Markdown'];
    if ($reply_to_message_id) $params['reply_to_message_id'] = $reply_to_message_id;
    if ($keyboard) $params['reply_markup'] = $keyboard;
    return telegram_request('sendPhoto', $params);
}

function sendVideo($chat_id, $video, $caption = null, $reply_to_message_id = null, $keyboard = null) {
    $params = ['chat_id' => $chat_id, 'video' => $video, 'caption' => $caption, 'parse_mode' => 'Markdown'];
    if ($reply_to_message_id) $params['reply_to_message_id'] = $reply_to_message_id;
    if ($keyboard) $params['reply_markup'] = $keyboard;
    return telegram_request('sendVideo', $params);
}

function forwardMessage($to_chat_id, $from_chat_id, $message_id) {
    return telegram_request('forwardMessage', ['chat_id' => $to_chat_id, 'from_chat_id' => $from_chat_id, 'message_id' => $message_id]);
}

/* ---------- STORAGE: JSON or TXT ---------- */
function ensure_files_exist() {
    if (USE_TXT_FILES) {
        if (!file_exists(USERS_FILE_TXT)) @file_put_contents(USERS_FILE_TXT, "", LOCK_EX);
        if (!file_exists(STATUS_FILE_TXT)) @file_put_contents(STATUS_FILE_TXT, "bot_active:1", LOCK_EX);
        if (!file_exists(STATE_FILE_TXT)) @file_put_contents(STATE_FILE_TXT, "", LOCK_EX);
    } else {
        if (!file_exists(USERS_FILE_JSON)) @file_put_contents(USERS_FILE_JSON, json_encode(new stdClass(), JSON_UNESCAPED_UNICODE), LOCK_EX);
        if (!file_exists(STATUS_FILE_JSON)) @file_put_contents(STATUS_FILE_JSON, json_encode(['bot_active' => true], JSON_UNESCAPED_UNICODE), LOCK_EX);
        if (!file_exists(STATE_FILE_JSON)) @file_put_contents(STATE_FILE_JSON, json_encode(new stdClass(), JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
}
ensure_files_exist();

function load_users() {
    if (USE_TXT_FILES) {
        $lines = @file(USERS_FILE_TXT, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $users = [];
        if ($lines) {
            foreach ($lines as $ln) {
                $parts = explode('|', trim($ln));
                if (count($parts) >= 2) $users[$parts[0]] = ['user_id' => (int)$parts[0], 'join_time' => (int)$parts[1]];
            }
        }
        return $users;
    } else {
        $c = @file_get_contents(USERS_FILE_JSON);
        $a = json_decode($c, true);
        return $a ?? [];
    }
}

function save_users($users) {
    if (USE_TXT_FILES) {
        $lines = [];
        foreach ($users as $u) $lines[] = $u['user_id'] . '|' . $u['join_time'];
        @file_put_contents(USERS_FILE_TXT, implode(PHP_EOL, $lines), LOCK_EX);
    } else {
        @file_put_contents(USERS_FILE_JSON, json_encode($users, JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
}

function load_status() {
    if (USE_TXT_FILES) {
        $c = @file_get_contents(STATUS_FILE_TXT);
        if ($c === false) return ['bot_active' => true];
        $c = trim($c);
        if (strpos($c, 'bot_active:') === 0) {
            $v = (int)substr($c, strlen('bot_active:'));
            return ['bot_active' => ($v === 1)];
        }
        return ['bot_active' => true];
    } else {
        $c = @file_get_contents(STATUS_FILE_JSON);
        $a = json_decode($c, true);
        return $a ?? ['bot_active' => true];
    }
}

function save_status($status) {
    if (USE_TXT_FILES) {
        $v = !empty($status['bot_active']) ? 1 : 0;
        @file_put_contents(STATUS_FILE_TXT, "bot_active:$v", LOCK_EX);
    } else {
        @file_put_contents(STATUS_FILE_JSON, json_encode($status, JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
}

function load_admin_state() {
    if (USE_TXT_FILES) {
        $lines = @file(STATE_FILE_TXT, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        $state = [];
        if ($lines) {
            foreach ($lines as $ln) {
                $j = json_decode($ln, true);
                if ($j && isset($j['id'])) $state[$j['id']] = $j['state'];
            }
        }
        return $state;
    } else {
        $c = @file_get_contents(STATE_FILE_JSON);
        $a = json_decode($c, true);
        return $a ?? [];
    }
}

function save_admin_state($state) {
    if (USE_TXT_FILES) {
        $lines = [];
        foreach ($state as $id => $s) $lines[] = json_encode(['id' => (int)$id, 'state' => $s], JSON_UNESCAPED_UNICODE);
        @file_put_contents(STATE_FILE_TXT, implode(PHP_EOL, $lines), LOCK_EX);
    } else {
        @file_put_contents(STATE_FILE_JSON, json_encode($state, JSON_UNESCAPED_UNICODE), LOCK_EX);
    }
}

/* ---------- HELPERS ---------- */
function get_main_keyboard($optional_channels = []) {
    $kb = [
        'inline_keyboard' => [
            [['text' => '📥 دانلود پست/ریلز', 'callback_data' => 'help_post'], ['text' => '📸 دانلود استوری', 'callback_data' => 'help_story']],
            [['text' => '👤 اطلاعات پیج', 'callback_data' => 'help_info'], ['text' => 'ℹ️ راهنما', 'callback_data' => 'help']],
            [['text' => '📞 پشتیبانی', 'url' => 'https://t.me/Adrianlamu']]
        ]
    ];
    // optional channels as a row of buttons
    if (!empty($optional_channels)) {
        $row = [];
        foreach ($optional_channels as $ch) {
            $btn = ['text' => (strpos($ch, '@') === 0 ? substr($ch,1) : $ch), 'url' => 'https://t.me/' . ltrim($ch, '@')];
            $row[] = $btn;
        }
        $kb['inline_keyboard'][] = $row;
    }
    return json_encode($kb);
}

function get_admin_keyboard() {
    return json_encode([
        'inline_keyboard' => [
            [['text' => '📊 آمار کاربران', 'callback_data' => 'admin_stats'], ['text' => '📢 پخش پیام', 'callback_data' => 'admin_broadcast']],
            [['text' => '🔄 پخش فوروارد', 'callback_data' => 'admin_forward'], ['text' => '🔌 روشن/خاموش ربات', 'callback_data' => 'admin_toggle']],
            [['text' => '🚪 خروج', 'callback_data' => 'exit_admin']]
        ]
    ]);
}

/* ---------- INSTAGRAM FETCH (همان رفتار قبلی) ---------- */
function fetchInstagramData($type, $value, $is_username = false) {
    $param_key = $is_username ? 'username' : 'url';
    $url = INSTAGRAM_API_URL . '?apikey=' . urlencode(API_KEY) . '&type=' . urlencode($type) . '&' . $param_key . '=' . urlencode($value);

    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_TIMEOUT, 60);
    $response = curl_exec($ch);
    $http = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if (curl_errno($ch)) {
        $err = curl_error($ch);
        debug_log("fetchInstagramData cURL error: $err url=$url http=$http");
        curl_close($ch);
        return ['ok' => false, 'error' => $err, 'http_code' => $http];
    }
    curl_close($ch);
    debug_log("fetchInstagramData http=$http url=$url body=" . $response);
    $data = json_decode($response, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        debug_log("fetchInstagramData json decode error: " . json_last_error_msg());
        return ['ok' => false, 'error' => 'JSON decode error', 'raw' => $response];
    }
    if (!isset($data['ok']) || $data['ok'] !== true || !isset($data['result'])) {
        $msg = $data['result']['message'] ?? $data['message'] ?? 'Unknown error';
        debug_log("fetchInstagramData reported error: $msg full=" . json_encode($data, JSON_UNESCAPED_UNICODE));
        return ['ok' => false, 'error' => $msg, 'raw' => $data];
    }
    return $data;
}

/* ---------- FORCE JOIN CHECK ---------- */
function is_user_member_of($chat_identifier, $user_id) {
    // chat_identifier can be username (without @) or numeric id
    $chat = $chat_identifier;
    if (is_numeric($chat_identifier)) $chat = (int)$chat_identifier;
    $resp = telegram_request('getChatMember', ['chat_id' => $chat, 'user_id' => $user_id]);
    debug_log("getChatMember for $chat user=$user_id resp=" . json_encode($resp, JSON_UNESCAPED_UNICODE));
    if (!isset($resp['ok']) || !$resp['ok']) return false;
    $status = $resp['result']['status'] ?? '';
    // statuses like 'left' or 'kicked' are not allowed
    return !in_array($status, ['left', 'kicked']);
}

/* ---------- MAIN WEBHOOK HANDLER ---------- */
$update = json_decode(file_get_contents('php://input'), true);
debug_log("incoming update: " . json_encode($update, JSON_UNESCAPED_UNICODE));

// handle callback_query and messages similar to original flow
$is_group = isset($update['message']['chat']['type']) && ($update['message']['chat']['type'] === 'group' || $update['message']['chat']['type'] === 'supergroup');

if (isset($update['callback_query'])) {
    $callback = $update['callback_query'];
    $chat_id = $callback['message']['chat']['id'];
    $message_id = $callback['message']['message_id'];
    $data = $callback['data'];
    $from_id = $callback['from']['id'];
    $is_admin = ($from_id == ADMIN_ID);

    if (strpos($data, 'admin_') === 0 && !$is_admin) {
        telegram_request('answerCallbackQuery', ['callback_query_id' => $callback['id'], 'text' => 'شما ادمین نیستید!']);
        exit;
    }

    $keyboard = $is_group ? null : get_main_keyboard($OPTIONAL_CHANNELS ?? []);

    switch ($data) {
        case 'help_post':
            sendMessage($chat_id, "📥 لینک پست یا ریلز را ارسال کنید:\nمثال: `https://www.instagram.com/p/...` یا `.../reel/...`", $keyboard);
            break;
        case 'help_story':
            sendMessage($chat_id, "📸 لینک استوری را ارسال کنید:\nمثال: `https://www.instagram.com/stories/...`", $keyboard);
            break;
        case 'help_info':
            sendMessage($chat_id, "👤 نام کاربری پیج را ارسال کنید (بدون @):\nمثال: `instagram`", $keyboard);
            break;
        case 'help':
            sendMessage($chat_id, "🌟 *راهنمای ربات* 🌟\n\n- پست/ریلز: لینک پست ارسال کنید\n- استوری: لینک استوری ارسال کنید\n- اطلاعات پیج: نام کاربری ارسال کنید\n\nپشتیبانی از آلبوم‌ها و محتوای چندرسانه‌ای.", $keyboard);
            break;
        case 'admin_stats':
            $users = load_users();
            $stats = get_user_stats($users);
            sendMessage($chat_id, "📊 *آمار*:\nکل: {$stats['total']}\nروزانه جدید: {$stats['daily_new']}\nهفتگی جدید: {$stats['weekly_new']}", get_admin_keyboard());
            break;
        case 'admin_broadcast':
            $state = load_admin_state();
            $state[$from_id] = 'broadcast';
            save_admin_state($state);
            sendMessage($chat_id, "📢 متن پیام را وارد کنید (لغو: /cancel):", get_admin_keyboard());
            break;
        case 'admin_forward':
            $state = load_admin_state();
            $state[$from_id] = 'forward';
            save_admin_state($state);
            sendMessage($chat_id, "🔄 پیام را فوروارد کنید (لغو: /cancel):", get_admin_keyboard());
            break;
        case 'admin_toggle':
            $status = load_status();
            $status['bot_active'] = !$status['bot_active'];
            save_status($status);
            $state_text = $status['bot_active'] ? 'روشن' : 'خاموش';
            sendMessage($chat_id, "🔌 ربات {$state_text} شد.", get_admin_keyboard());
            break;
        case 'exit_admin':
            deleteMessage($chat_id, $message_id);
            sendMessage($chat_id, "🚪 خارج شدید.", $keyboard);
            break;
    }

    telegram_request('answerCallbackQuery', ['callback_query_id' => $callback['id']]);
    exit;
}

if (!isset($update['message'])) {
    http_response_code(200);
    exit;
}

$message = $update['message'];
$chat_id = $message['chat']['id'];
$from_id = $message['from']['id'];
$text = trim($message['text'] ?? '');
$reply_to_message_id = $message['message_id'];
$is_forwarded = isset($message['forward_from']) || isset($message['forward_from_chat']);

// mentions detection
$mentioned = false;
if (isset($message['entities']) && is_array($message['entities'])) {
    foreach ($message['entities'] as $entity) {
        if (isset($entity['type']) && ($entity['type'] === 'mention' || $entity['type'] === 'text_mention')) {
            $mentioned = true;
            break;
        }
    }
}
$replied_to_bot = isset($message['reply_to_message']) && isset($message['reply_to_message']['from']['is_bot']) && $message['reply_to_message']['from']['is_bot'];

if ($is_group && !$mentioned && !$replied_to_bot && $text !== '/start' && $text !== '/admin') {
    http_response_code(200);
    exit;
}

/* ---------- FORCE JOIN verification ---------- */
global $REQUIRED_CHANNELS, $OPTIONAL_CHANNELS;
if (FORCE_JOIN && !empty($REQUIRED_CHANNELS)) {
    foreach ($REQUIRED_CHANNELS as $ch) {
        // if any required channel not member, send message and stop
        if (!is_user_member_of($ch, $from_id)) {
            $keyboard = json_encode(['inline_keyboard' => [[['text' => 'عضویت در کانال', 'url' => 'https://t.me/' . ltrim($ch,'@')]], [['text' => 'برگشت', 'callback_data' => 'help']]]]);
            sendMessage($chat_id, "❗ برای استفاده از ربات باید عضو کانال @{ch} شوید. سپس /start را ارسال کنید.", $keyboard, $reply_to_message_id);
            http_response_code(200);
            exit;
        }
    }
}

/* ---------- bot active check ---------- */
$status = load_status();
if (!$status['bot_active'] && $from_id != ADMIN_ID) {
    sendMessage($chat_id, "🤖 ربات خاموش است. بعداً امتحان کنید.", null, $reply_to_message_id);
    exit;
}

/* ---------- register user ---------- */
$users = load_users();
if (!array_key_exists($from_id, $users)) {
    $users[$from_id] = ['user_id' => $from_id, 'join_time' => time()];
    save_users($users);
}

$admin_state = load_admin_state();
$current_state = $admin_state[$from_id] ?? null;

/* ---------- admin handling ---------- */
if ($from_id == ADMIN_ID && $current_state && $text !== '/cancel') {
    if ($current_state === 'broadcast' && !$is_forwarded) {
        $success = broadcast_message($text);
        sendMessage($chat_id, "📢 ارسال به {$success} کاربر موفق.", get_admin_keyboard());
        unset($admin_state[$from_id]);
        save_admin_state($admin_state);
        exit;
    } elseif ($current_state === 'forward' && $is_forwarded) {
        $from_chat_id = $message['forward_from_chat']['id'] ?? $message['forward_from']['id'];
        $orig_message_id = $message['forward_from_message_id'] ?? $message['message_id'];
        $success = broadcast_message(null, true, $from_chat_id, $orig_message_id);
        sendMessage($chat_id, "🔄 فوروارد به {$success} کاربر موفق.", get_admin_keyboard());
        unset($admin_state[$from_id]);
        save_admin_state($admin_state);
        exit;
    }
}

/* ---------- keyboard with optional channels if FORCE_JOIN=false ---------- */
$keyboard = $is_group ? null : get_main_keyboard(FORCE_JOIN ? [] : $OPTIONAL_CHANNELS);

/* ---------- basic commands ---------- */
if ($text === '/start') {
    $start_message = "🌟 *دانلودر اینستاگرام پیشرفته* 🌟\n\nدانلود پست، ریلز، استوری و اطلاعات پیج.\nلینک یا نام کاربری ارسال کنید!\n\n✅ *نمونه‌ها*:\n- پست: `https://www.instagram.com/p/...`\n- استوری: `https://www.instagram.com/stories/...`\n- پیج: `instagram`";
    sendMessage($chat_id, $start_message, $keyboard, $reply_to_message_id);
    exit;
} elseif ($text === '/admin' && $from_id == ADMIN_ID) {
    sendMessage($chat_id, "🛡️ *پنل ادمین*", get_admin_keyboard());
    exit;
} elseif ($text === '/cancel' && $from_id == ADMIN_ID && $current_state) {
    unset($admin_state[$from_id]);
    save_admin_state($admin_state);
    sendMessage($chat_id, "❌ لغو شد.", get_admin_keyboard());
    exit;
}

/* ---------- processing user input (same logic as before) ---------- */
$temp_message = sendMessage($chat_id, "⏳ پردازش...", null, $reply_to_message_id);
$temp_message_id = $temp_message['result']['message_id'] ?? null;

$input_value = $text;
$is_link = filter_var($input_value, FILTER_VALIDATE_URL);
$types_to_try = [];
$api_response = null;
$is_username = false;

if (!$is_link) {
    if (preg_match('/^[a-zA-Z0-9._]+$/', $input_value)) {
        $types_to_try = ['info'];
        $is_username = true;
    } else {
        if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
        sendMessage($chat_id, "❌ ورودی نامعتبر. لطفاً لینک یا نام کاربری صحیح ارسال کنید.", null, $reply_to_message_id);
        exit;
    }
} else {
    if (preg_match('/instagram\.com\/(p\/|reel\/)/i', $input_value)) {
        $types_to_try = ['post2', 'post'];
    } elseif (preg_match('/instagram\.com\/stories\//i', $input_value)) {
        $types_to_try = ['story2', 'story'];
    } else {
        if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
        sendMessage($chat_id, "❌ نوع لینک پشتیبانی نمی‌شود.", null, $reply_to_message_id);
        exit;
    }
}

foreach ($types_to_try as $current_type) {
    $api_response = fetchInstagramData($current_type, $input_value, $is_username);
    if (isset($api_response['ok']) && $api_response['ok']) break;
}

if (!$api_response || !(isset($api_response['ok']) && $api_response['ok'])) {
    $err = $api_response['error'] ?? 'نامشخص';
    if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
    sendMessage($chat_id, "❌ خطا: " . $err, null, $reply_to_message_id);
    exit;
}

$raw_results = $api_response['result'] ?? null;
$results = $raw_results['result'] ?? $raw_results ?? [];

if (empty($results)) {
    if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
    sendMessage($chat_id, "❌ نتیجه‌ای یافت نشد.", null, $reply_to_message_id);
    exit;
}

/* ---------- info handling ---------- */
if ($types_to_try[0] === 'info') {
    $info = $results[0] ?? $results;
    $caption = "👤 *پیج {$input_value}*:\n" .
               "نام: " . ($info['full_name'] ?? 'نامشخص') . "\n" .
               "بیو: " . ($info['bio'] ?? 'نامشخص') . "\n" .
               "فالوور: " . ($info['followers'] ?? 'نامشخص') . "\n" .
               "فالوئینگ: " . ($info['following'] ?? 'نامشخص') . "\n" .
               "پست‌ها: " . ($info['posts'] ?? 'نامشخص');

    $profile_pic = $info['profile_pic_url'] ?? null;
    if ($profile_pic) {
        sendPhoto($chat_id, $profile_pic, $caption, $reply_to_message_id, null);
    } else {
        sendMessage($chat_id, $caption, null, $reply_to_message_id);
    }
    if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
    exit;
}

/* ---------- media handling (single/album) ---------- */
$first_item = $results[0];
$caption = '';
$raw_caption = $first_item['caption'] ?? '';
if (!empty($raw_caption)) {
    $cleaned_caption = html_entity_decode($raw_caption, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    $caption_prefix = "✍️ *کپشن*:\n";
    $caption_text = mb_strlen($cleaned_caption, 'UTF-8') > 800 ? mb_substr($cleaned_caption, 0, 800, 'UTF-8') . '...' : $cleaned_caption;
    $caption = $caption_prefix . $caption_text;
}

$is_album = isset($first_item['is_album']) && $first_item['is_album'] === true && count($results) > 1;

if ($is_album) {
    $all_media_items = array_slice($results, 1);
    $total_items = count($all_media_items);
    $chunk_size = 10;
    $total_chunks = ceil($total_items / $chunk_size);
    $successful_sends = 0;

    for ($chunk_index = 0; $chunk_index < $total_chunks; $chunk_index++) {
        $start_index = $chunk_index * $chunk_size;
        $media_chunk = array_slice($all_media_items, $start_index, $chunk_size);
        if (empty($media_chunk)) continue;

        $media_group = [];
        $item_in_chunk_count = 0;

        $part_text = $total_chunks > 1 ? "\n\n*" . ($chunk_index + 1) . " از " . $total_chunks . "*" : "";
        $current_caption = !empty($caption) ? $caption . $part_text : trim($part_text);

        foreach ($media_chunk as $item) {
            $media_url = $item['video_url'] ?? $item['image_url'] ?? $item['url'] ?? null;
            if (!$media_url) continue;
            $media_type = ($item['is_video'] ?? false) || ($item['type'] ?? '') === 'video' ? 'video' : 'photo';
            $input_media = ['type' => $media_type, 'media' => $media_url];
            if ($item_in_chunk_count === 0 && !empty($current_caption)) $input_media['caption'] = $current_caption;
            $media_group[] = $input_media;
            $item_in_chunk_count++;
        }

        if (!empty($media_group)) {
            $response = sendMediaGroup($chat_id, $media_group, $chunk_index === 0 ? $reply_to_message_id : null);
            if (isset($response['ok']) && $response['ok'] === true) $successful_sends++;
            else debug_log("sendMediaGroup failed: " . json_encode($response, JSON_UNESCAPED_UNICODE));
        }
    }

    if ($successful_sends === 0) {
        if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
        sendMessage($chat_id, "❌ ارسال آلبوم ناموفق بود.", null, $reply_to_message_id);
        exit;
    }

} else {
    $media_item = $results[0];
    if (!isset($media_item['video_url']) && !isset($media_item['image_url']) && !isset($media_item['url']) && isset($results[1])) {
        $media_item = $results[1];
    }
    if (!$media_item) {
        if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
        sendMessage($chat_id, "❌ محتوایی یافت نشد.", null, $reply_to_message_id);
        exit;
    }

    $media_url = $media_item['video_url'] ?? $media_item['image_url'] ?? $media_item['url'] ?? null;
    $is_video = ($media_item['is_video'] ?? false) || ($media_item['type'] ?? '') === 'video';

    if (!$media_url) {
        if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
        sendMessage($chat_id, "❌ آدرس رسانه موجود نیست.", null, $reply_to_message_id);
        exit;
    } else {
        $response = $is_video ? sendVideo($chat_id, $media_url, $caption, $reply_to_message_id, null) : sendPhoto($chat_id, $media_url, $caption, $reply_to_message_id, null);
        if (!$response || ($response['ok'] ?? false) !== true) {
            debug_log("sendPhoto/Video failed: " . json_encode($response, JSON_UNESCAPED_UNICODE));
            if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
            sendMessage($chat_id, "❌ ارسال رسانه با خطا مواجه شد.", null, $reply_to_message_id);
            exit;
        }
    }
}

if ($temp_message_id) deleteMessage($chat_id, $temp_message_id);
http_response_code(200);

/* ---------- helper functions used earlier ---------- */
function get_user_stats($users) {
    $total_users = count($users);
    $now = time();
    $daily_new = 0;
    $weekly_new = 0;
    foreach ($users as $user) {
        $join_time = $user['join_time'];
        if ($now - $join_time < 86400) $daily_new++;
        if ($now - $join_time < 604800) $weekly_new++;
    }
    return ['total' => $total_users, 'daily_new' => $daily_new, 'weekly_new' => $weekly_new];
}

function broadcast_message($text, $is_forward = false, $from_chat_id = null, $message_id = null) {
    $users = load_users();
    $success_count = 0;
    foreach ($users as $user) {
        $chat_id = $user['user_id'];
        if ($is_forward && $from_chat_id && $message_id) {
            $response = forwardMessage($chat_id, $from_chat_id, $message_id);
        } else {
            $response = sendMessage($chat_id, $text);
        }
        if (isset($response['ok']) && $response['ok']) $success_count++;
        usleep(50000);
    }
    return $success_count;
}

?>
